// content.js - V15 (注册/登录/JWT安全验证/7天免登/完整功能)

// --- 配置 ---
const SERVER_URL = "https://tiktokus-mail.onrender.com"; 

// --- 样式定义 ---
const STYLE = `
<style>
  * { box-sizing: border-box; }
  #wonder-launcher {
    position: fixed; bottom: 20px; right: 20px; width: 50px; height: 50px;
    background: #000; border-radius: 50%; box-shadow: 0 4px 15px rgba(0,0,0,0.2);
    cursor: pointer; z-index: 2147483647; display: flex; align-items: center; justify-content: center;
    color: white; font-weight: bold; font-size: 20px; user-select: none; font-family: sans-serif;
    transition: transform 0.2s;
  }
  #wonder-launcher:hover { transform: scale(1.1); }
  
  #wonder-panel-root {
    position: fixed; bottom: 80px; right: 20px; width: 320px; height: 540px;
    background: rgba(255, 255, 255, 0.98); border-radius: 12px;
    box-shadow: 0 10px 40px rgba(0,0,0,0.2); z-index: 2147483646;
    overflow: hidden; display: none; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    animation: slideUp 0.3s ease; border: 1px solid rgba(0,0,0,0.05);
  }

  .slide-container { display: flex; width: 200%; height: 100%; transition: transform 0.3s cubic-bezier(0.25, 0.8, 0.25, 1); }
  .slide-page { width: 50%; height: 100%; display: flex; flex-direction: column; position: relative; }

  /* 头部样式 */
  .wonder-header { padding: 0 16px; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; align-items: center; background: #fff; height: 50px; flex-shrink: 0; }
  .header-left { display: flex; align-items: center; gap: 8px; }
  .header-title { font-weight: 700; font-size: 15px; color: #333; }
  .header-actions { display: flex; gap: 12px; align-items: center; }
  
  .logout-text-btn { font-size: 12px; color: #999; cursor: pointer; text-decoration: none; }
  .logout-text-btn:hover { color: #666; text-decoration: underline; }

  .icon-btn { width: 28px; height: 28px; border-radius: 4px; display: flex; align-items: center; justify-content: center; cursor: pointer; color: #666; font-size: 14px; transition: background 0.2s; }
  .icon-btn:hover { background: #f0f0f0; }

  /* 内容区域 */
  .wonder-body { padding: 16px; flex: 1; overflow-y: auto; display: flex; flex-direction: column; }
  
  .w-label { font-size: 12px; color: #888; margin-bottom: 4px; display: block; }
  .w-input { width: 100%; padding: 8px 10px; margin-bottom: 12px; border: 1px solid #eee; border-radius: 6px; font-size: 13px; outline: none; background: #f9f9f9; }
  .w-input:focus { border-color: #007AFF; background: #fff; }
  
  .platform-badge { font-size: 10px; padding: 2px 6px; border-radius: 4px; color: white; margin-bottom: 10px; display: inline-block; font-weight: 600; }
  .pf-tiktok { background: #000; }
  .pf-instagram { background: linear-gradient(45deg, #f09433, #e6683c, #dc2743, #cc2366, #bc1888, #8a3ab9); }
  .pf-youtube { background: #FF0000; }
  .pf-other { background: #999; }

  .w-btn { width: 100%; padding: 10px; background: #007AFF; color: white; border: none; border-radius: 8px; cursor: pointer; font-weight: 500; font-size: 13px; transition: 0.2s; text-align: center; }
  .w-btn:hover { background: #0062cc; }
  
  .col-row { display: flex; gap: 8px; margin-bottom: 12px; align-items: center; }
  .add-col-btn { width: 28px; height: 36px; background: #eee; color: #666; border: none; border-radius: 6px; cursor: pointer; font-size: 16px; display: flex; align-items: center; justify-content: center; }
  
  .uid-text { font-size: 10px; color: #aaa; margin-top: -8px; margin-bottom: 10px; margin-left: 2px; }
  
  /* 清单列表样式 */
  .list-toolbar { padding: 10px 12px; background: #f9f9f9; border-bottom: 1px solid #eee; display: flex; flex-direction: column; gap: 8px; flex-shrink: 0; }
  .search-row { display: flex; gap: 5px; }
  .search-input { flex: 1; padding: 6px; border: 1px solid #ddd; border-radius: 4px; font-size: 12px; outline: none; }
  
  .action-row { display: flex; justify-content: space-between; align-items: center; }
  .filter-row { display: flex; align-items: center; gap: 8px; font-size: 11px; color: #666; }
  .export-group { display: flex; gap: 5px; }
  .xs-btn { font-size: 10px; padding: 4px 8px; border-radius: 4px; border: 1px solid #ddd; background: #fff; cursor: pointer; color: #555; }
  .xs-btn:hover { background: #eee; }

  .lead-item { padding: 12px; border-bottom: 1px solid #f5f5f5; font-size: 13px; display: flex; justify-content: space-between; align-items: center; transition: background 0.1s; }
  .lead-item:hover { background: #fcfcfc; }
  
  .lead-left { display: flex; flex-direction: column; gap: 3px; max-width: 75%; }
  .lead-name { font-weight: 500; color: #333; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; }
  .lead-meta { font-size: 10px; color: #999; }
  .lead-email { color: #007AFF; font-weight: 500; }

  .lead-right { display: flex; flex-direction: column; align-items: flex-end; gap: 4px; }
  .lead-status { font-size: 10px; color: #aaa; }
  .lead-status.sent { color: #34c759; }
  
  .btn-delete { width: 24px; height: 24px; display: flex; align-items: center; justify-content: center; cursor: pointer; color: #ddd; border-radius: 4px; transition: all 0.2s; font-size: 14px; }
  .btn-delete:hover { background: #ffeaea; color: #ff3b30; }
  .btn-delete.disabled { cursor: not-allowed; opacity: 0.3; }

  .duplicate-tag { color: #ff3b30; font-size: 10px; margin-left: 5px; font-weight: bold; }
  .bulk-area { width: 100%; height: 100px; border: 1px solid #eee; border-radius: 6px; padding: 8px; font-size: 11px; resize: none; margin-bottom: 10px; }

  /* 底部统计栏 */
  .list-footer { height: 30px; background: #f1f1f1; border-top: 1px solid #ddd; display: flex; align-items: center; justify-content: center; font-size: 10px; color: #666; gap: 10px; flex-shrink: 0; }
  .stat-item { display: flex; align-items: center; gap: 3px; }
  .dot { width: 6px; height: 6px; border-radius: 50%; display: inline-block; }

  /* 登录/注册专用样式 */
  .auth-switch { text-align: center; margin-top: 15px; font-size: 12px; color: #666; }
  .auth-switch a { color: #007AFF; cursor: pointer; text-decoration: underline; }
  .verify-row { display: flex; gap: 8px; }
  .btn-code { width: 90px; padding: 8px; font-size: 11px; background: #eee; color: #333; border:none; border-radius:6px; cursor:pointer; }
</style>
`;

// --- HTML 模板 ---
const UI_TEMPLATE = `
  ${STYLE}
  <div id="wonder-launcher" style="display:none;">L</div>
  <div id="wonder-panel-root">
    <div class="slide-container" id="slider">
        
        <div class="slide-page">
            <div class="wonder-header">
                <div class="header-left">
                    <span class="header-title">LeadTracking</span>
                </div>
                <div class="header-actions">
                    <span class="logout-text-btn" id="btn-logout">退出登录</span>
                    <div class="icon-btn" id="btn-minimize" title="最小化">－</div>
                </div>
            </div>
            
            <div id="view-login" class="wonder-body" style="display:none;">
                <div style="text-align:center; margin: 30px 0 20px;">👋 欢迎回来</div>
                <input type="text" id="inp-login-user" class="w-input" placeholder="用户名/邮箱">
                <input type="password" id="inp-login-pass" class="w-input" placeholder="密码">
                <button id="btn-login" class="w-btn">登录</button>
                <div class="auth-switch">
                    没有账号? <a id="link-to-register">去注册</a>
                </div>
            </div>

            <div id="view-register" class="wonder-body" style="display:none;">
                <div style="text-align:center; margin: 20px 0 15px;">📝 新用户注册</div>
                <input type="text" id="inp-reg-user" class="w-input" placeholder="设置用户名">
                <div class="verify-row">
                    <input type="text" id="inp-reg-email" class="w-input" placeholder="邮箱地址">
                    <button id="btn-send-code" class="btn-code">发送验证码</button>
                </div>
                <input type="text" id="inp-reg-code" class="w-input" placeholder="验证码 (6位)">
                <input type="password" id="inp-reg-pass" class="w-input" placeholder="设置密码">
                <button id="btn-register" class="w-btn" style="background:#34c759;">注册并登录</button>
                <div class="auth-switch">
                    已有账号? <a id="link-to-login">去登录</a>
                </div>
            </div>

            <div id="view-main" class="wonder-body" style="display:none;">
                <div style="flex:1;">
                    <div id="platform-badge" class="platform-badge pf-other">Other</div>
                    
                    <label class="w-label">选择集合</label>
                    <div class="col-row" style="margin-bottom:2px;">
                        <select id="sel-collection" class="w-input" style="margin-bottom:0; flex:1;"></select>
                        <button id="btn-add-col" class="add-col-btn" title="新建集合">+</button>
                    </div>
                    <div id="col-count-display" style="font-size:10px; color:#999; margin-bottom:12px; text-align:right; padding-right:2px;">加载中...</div>

                    <div style="text-align:right; margin-bottom:15px;">
                        <a href="#" id="link-view-list" style="font-size:11px; color:#007AFF; text-decoration:none;">查看清单 & 导入 ></a>
                    </div>

                    <label class="w-label">达人名称 <span id="dup-warning" class="duplicate-tag" style="display:none;">(已存在)</span></label>
                    <input type="text" id="inp-influencer" class="w-input">
                    <div id="txt-uid" class="uid-text">UID: 识别中...</div>

                    <label class="w-label">邮箱</label>
                    <input type="text" id="inp-email" class="w-input" placeholder="输入邮箱...">

                    <button id="btn-save" class="w-btn" style="margin-top:10px;">保存线索</button>
                    <div id="msg-box" style="margin-top:10px; font-size:12px; text-align:center; color:#34c759; height:20px;"></div>
                </div>
            </div>
        </div>

        <div class="slide-page" style="background:#fff;">
            <div class="wonder-header">
                <span class="header-title" id="list-title">集合清单</span>
                <div class="icon-btn" id="btn-back">←</div>
            </div>
            
            <div class="list-toolbar">
                <div class="search-row">
                    <input type="text" id="inp-list-search" class="search-input" placeholder="搜索达人名字或UID...">
                </div>
                <div style="display:flex; justify-content:space-between; align-items:center; margin-top:5px;">
                     <span id="list-count" style="font-size:10px; color:#999;">加载中...</span>
                </div>
                <div class="action-row">
                    <div class="filter-row">
                        <div>
                            <input type="checkbox" id="chk-today-only" style="vertical-align:middle;">
                            <label for="chk-today-only" style="cursor:pointer; vertical-align:middle;">仅今天</label>
                        </div>
                        <div>
                            <input type="checkbox" id="chk-unsent-only" style="vertical-align:middle;">
                            <label for="chk-unsent-only" style="cursor:pointer; vertical-align:middle;">仅未发信</label>
                        </div>
                    </div>
                    <div class="export-group">
                        <button class="xs-btn" id="btn-toggle-import">导入</button>
                        <button class="xs-btn" id="btn-export-exec">导出全部</button>
                    </div>
                </div>
            </div>

            <div id="import-box" style="padding:10px; border-bottom:1px solid #eee; display:none; background:#f9f9f9;">
                <div style="font-size:10px; color:#666; margin-bottom:5px;">格式: 名字,邮箱 (一行一个)</div>
                <textarea id="inp-bulk" class="bulk-area" placeholder="example_user, test@mail.com"></textarea>
                <button id="btn-do-import" class="w-btn" style="height:30px; line-height:10px;">执行导入</button>
            </div>
            
            <div id="lead-list" style="flex:1; overflow-y:auto;"></div>

            <div class="list-footer" id="list-footer"></div>
        </div>
    </div>
  </div>
`;

// --- 初始化 DOM ---
const WRAPPER_ID = 'wonder-extension-wrapper';
if (document.getElementById(WRAPPER_ID)) document.getElementById(WRAPPER_ID).remove();
const wrapper = document.createElement('div');
wrapper.id = WRAPPER_ID;
wrapper.innerHTML = UI_TEMPLATE;
document.body.appendChild(wrapper);

// --- 变量 ---
const launcher = document.getElementById('wonder-launcher');
const panel = document.getElementById('wonder-panel-root');
const slider = document.getElementById('slider');
let currentUser = null;
let currentToken = null; // 全局 Token
let currentPlatform = 'Other';
let currentUid = '';
let currentCollectionList = []; 
let filteredList = [];          
let isPanelOpen = false;

// --- 启动 ---
init();

function init() {
    detectPlatform(); // 初始抓取
    
    if (['TikTok', 'Instagram', 'YouTube'].includes(currentPlatform)) {
        launcher.style.display = 'flex';
    }

    const savedState = localStorage.getItem('wonderPanelOpen');
    if (savedState === 'true') togglePanel(true);

    // [核心安全检查] 检查 Token 有效性 (7天)
    chrome.storage.local.get(['user', 'token', 'tokenTime'], (res) => {
        const sevenDays = 604800000; // 7 * 24 * 60 * 60 * 1000
        const now = Date.now();

        if (res.token && res.user && res.tokenTime && (now - res.tokenTime < sevenDays)) {
            // Token 有效，自动登录
            currentUser = res.user;
            currentToken = res.token;
            switchToMainView();
        } else {
            // Token 过期或不存在，清理并显示登录页
            chrome.storage.local.remove(['user', 'token', 'tokenTime']);
            switchToLoginView();
        }
    });

    bindEvents();
}

// --- 1. 抓取逻辑 (包含 V13 YouTube 增强版) ---
function updateUidUI(uid) {
    currentUid = uid;
    document.getElementById('txt-uid').innerText = uid ? `UID: ${uid}` : 'UID: 未找到';
}

function detectPlatform() {
    const host = location.host;
    const badge = document.getElementById('platform-badge');
    
    if (host.includes('tiktok.com')) {
        currentPlatform = 'TikTok';
        badge.className = 'platform-badge pf-tiktok';
        scrapeTikTok();
    } else if (host.includes('instagram.com')) {
        currentPlatform = 'Instagram';
        badge.className = 'platform-badge pf-instagram';
        scrapeInstagram();
    } else if (host.includes('youtube.com')) {
        currentPlatform = 'YouTube';
        badge.className = 'platform-badge pf-youtube';
        scrapeYouTube();
    } else {
        currentPlatform = 'Other';
        badge.className = 'platform-badge pf-other';
    }
    badge.innerText = currentPlatform;
}

function scrapeInstagram() {
    const pathParts = location.pathname.split('/').filter(p => p);
    if (pathParts.length > 0) document.getElementById('inp-influencer').value = pathParts[0];
    let uidFound = null;
    try {
        const meta = document.querySelector('meta[property="al:ios:url"]');
        if (meta && meta.content.includes('id=')) uidFound = meta.content.split('id=')[1];
        if(!uidFound) {
            const match = document.body.innerHTML.match(/"user_id":"(\d+)"/);
            if(match) uidFound = match[1];
        }
    } catch(e){}
    updateUidUI(uidFound);
}

function scrapeTikTok() {
    const path = location.pathname;
    if (path.startsWith('/@')) document.getElementById('inp-influencer').value = path.substring(1).split('/')[0];
    let uid = null;
    try {
        const script = document.getElementById('__UNIVERSAL_DATA_FOR_REHYDRATION__');
        if (script) {
            const data = JSON.parse(script.textContent);
            const scope = data.__DEFAULT_SCOPE__;
            if(scope?.['webapp.user-detail']?.userInfo?.user?.id) uid = scope['webapp.user-detail'].userInfo.user.id;
        }
        if (!uid) {
            const match = document.body.innerHTML.match(/"authorId":"(\d+)"/) || document.body.innerHTML.match(/"ownerId":"(\d+)"/);
            if (match) uid = match[1];
        }
    } catch(e) {}
    updateUidUI(uid);
}

function scrapeYouTube() {
    const path = location.pathname;
    let foundName = '';
    let uid = '';

    const metaChannel = document.querySelector('meta[itemprop="channelId"]');
    const metaId = document.querySelector('meta[itemprop="identifier"]');
    if (metaChannel) uid = metaChannel.content;
    else if (metaId) uid = metaId.content;

    if (!uid && path.startsWith('/channel/')) {
        const parts = path.split('/');
        if (parts[2]) uid = parts[2];
    }

    if (path.startsWith('/@')) {
         const cleanPath = path.split('/')[1]; 
         foundName = cleanPath.substring(1); 
    } else {
        // 策略 1: 扫描 Header (适配新版布局)
        const metadataRows = document.querySelectorAll('yt-content-metadata-view-model span');
        for (let span of metadataRows) {
            const txt = span.innerText.trim();
            if (txt.startsWith('@')) {
                foundName = txt.replace(/^@/, '');
                break;
            }
        }
        // 策略 2: 扫描旧版 ID
        if (!foundName) {
            const handleEl = document.querySelector('#channel-handle');
            if (handleEl) foundName = handleEl.innerText.trim().replace(/^@/, '');
        }
        
        // 策略 3: 核武器 - 扫描页面脚本 (ytInitialData) 
        if (!foundName) {
            try {
                const scripts = document.querySelectorAll('script');
                const regex = /"canonicalBaseUrl":"\/@([^"]+)"/;
                for (let s of scripts) {
                    if (s.innerText.includes('canonicalBaseUrl')) {
                        const match = s.innerText.match(regex);
                        if (match && match[1]) {
                            foundName = match[1];
                            break;
                        }
                    }
                }
            } catch(e) {}
        }
    }

    if (foundName) document.getElementById('inp-influencer').value = foundName;
    updateUidUI(uid);
}

// --- 2. 交互逻辑 (事件绑定) ---

function togglePanel(show) {
    isPanelOpen = show;
    panel.style.display = show ? 'block' : 'none';
    localStorage.setItem('wonderPanelOpen', show);
    
    if(show) {
        detectPlatform();
        if(slider.style.transform === 'translateX(0px)' && currentToken) {
            checkDuplicate();
        }
    }
}

function bindEvents() {
    launcher.onclick = (e) => { e.stopPropagation(); togglePanel(true); };
    document.getElementById('btn-minimize').onclick = () => togglePanel(false);
    
    // --- 认证相关事件 ---
    document.getElementById('link-to-register').onclick = () => {
        document.getElementById('view-login').style.display = 'none';
        document.getElementById('view-register').style.display = 'block';
    };
    document.getElementById('link-to-login').onclick = () => {
        document.getElementById('view-register').style.display = 'none';
        document.getElementById('view-login').style.display = 'block';
    };

    document.getElementById('btn-send-code').onclick = handleSendCode;
    document.getElementById('btn-register').onclick = handleRegister;
    document.getElementById('btn-login').onclick = handleLogin;
    
    document.getElementById('inp-login-pass').addEventListener('keydown', (e) => {
        if(e.key === 'Enter') handleLogin();
    });

    document.getElementById('btn-logout').onclick = () => {
        if(confirm('确定退出登录吗?')) {
            currentUser = null;
            currentToken = null;
            chrome.storage.local.remove(['user', 'token', 'tokenTime']);
            localStorage.removeItem('wonderLastCollectionId');
            switchToLoginView();
        }
    };

    // --- 业务功能事件 ---
    document.getElementById('btn-save').onclick = handleSave;

    document.getElementById('link-view-list').onclick = (e) => {
        e.preventDefault(); 
        slider.style.transform = 'translateX(-50%)';
        fetchCollectionList();
    };

    document.getElementById('btn-back').onclick = () => {
        slider.style.transform = 'translateX(0)';
    };

    document.getElementById('sel-collection').onchange = (e) => {
        localStorage.setItem('wonderLastCollectionId', e.target.value);
        checkDuplicate();
        fetchCollectionList(); 
    };

    document.getElementById('btn-add-col').onclick = createCollection;

    document.getElementById('inp-list-search').oninput = triggerFilter;
    document.getElementById('chk-today-only').onchange = triggerFilter;
    document.getElementById('chk-unsent-only').onchange = triggerFilter;

    document.getElementById('btn-export-exec').onclick = handleExportCsv;

    document.getElementById('btn-toggle-import').onclick = () => {
        const box = document.getElementById('import-box');
        box.style.display = box.style.display === 'none' ? 'block' : 'none';
    };
    document.getElementById('btn-do-import').onclick = handleBulkImport;
}

// --- 3. 核心认证功能实现 ---

async function handleSendCode() {
    const email = document.getElementById('inp-reg-email').value;
    if(!email || !email.includes('@')) return alert('请输入有效邮箱');
    
    const btn = document.getElementById('btn-send-code');
    btn.disabled = true;
    btn.innerText = '发送中...';
    
    try {
        const res = await fetch(`${SERVER_URL}/api/send-code`, {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ email })
        });
        const data = await res.json();
        if(data.success) {
            alert('验证码已发送至邮箱 (有效期10分钟)');
            // 倒计时
            let count = 60;
            const timer = setInterval(() => {
                btn.innerText = `${count}s`;
                count--;
                if(count < 0) {
                    clearInterval(timer);
                    btn.innerText = '发送验证码';
                    btn.disabled = false;
                }
            }, 1000);
        } else {
            alert(data.error || '发送失败');
            btn.innerText = '发送验证码';
            btn.disabled = false;
        }
    } catch(e) {
        alert('网络错误');
        btn.innerText = '发送验证码';
        btn.disabled = false;
    }
}

async function handleRegister() {
    const username = document.getElementById('inp-reg-user').value;
    const email = document.getElementById('inp-reg-email').value;
    const code = document.getElementById('inp-reg-code').value;
    const password = document.getElementById('inp-reg-pass').value;

    if(!username || !email || !code || !password) return alert('请填写完整信息');

    const btn = document.getElementById('btn-register');
    btn.innerText = '注册中...';

    try {
        const res = await fetch(`${SERVER_URL}/api/register`, {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ username, email, password, code })
        });
        const data = await res.json();
        if(data.success) {
            alert('注册成功！请登录');
            document.getElementById('link-to-login').click();
            // 自动填充用户名
            document.getElementById('inp-login-user').value = username;
        } else {
            alert(data.error);
        }
    } catch(e) { alert('注册请求失败'); }
    finally { btn.innerText = '注册并登录'; }
}

async function handleLogin() {
    const u = document.getElementById('inp-login-user').value;
    const p = document.getElementById('inp-login-pass').value;
    if(!u || !p) return alert('请输入账号密码');
    
    const btn = document.getElementById('btn-login');
    btn.innerText = '登录中...';
    try {
        const res = await fetch(`${SERVER_URL}/api/login`, {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ username: u, password: p })
        });
        const data = await res.json();
        if(data.success) {
            currentUser = data.user;
            currentToken = data.token;
            // 存储 Token 和 时间戳
            chrome.storage.local.set({ 
                user: data.user, 
                token: data.token,
                tokenTime: Date.now() 
            });
            switchToMainView();
        } else { alert(data.error || '登录失败'); }
    } catch(e) { alert('网络错误'); } 
    finally { btn.innerText = '登录'; }
}

// [核心] 安全请求封装函数 (所有业务请求都必须走这个)
// 自动添加 Token，如果 401/403 则自动踢出
async function authFetch(url, options = {}) {
    if (!options.headers) options.headers = {};
    if (!currentToken) {
        alert('请先登录');
        switchToLoginView();
        return null;
    }
    options.headers['Authorization'] = `Bearer ${currentToken}`;
    options.headers['Content-Type'] = 'application/json';
    
    const res = await fetch(url, options);
    if (res.status === 401 || res.status === 403) {
        alert('登录已过期，请重新登录');
        currentUser = null;
        currentToken = null;
        chrome.storage.local.remove(['user', 'token', 'tokenTime']);
        switchToLoginView();
        return null; // 中断后续操作
    }
    return res;
}

// --- 视图切换 ---

function switchToMainView() {
    document.getElementById('view-login').style.display = 'none';
    document.getElementById('view-register').style.display = 'none';
    document.getElementById('view-main').style.display = 'flex';
    document.getElementById('view-main').style.flexDirection = 'column';
    loadCollections();
}

function switchToLoginView() {
    document.getElementById('view-login').style.display = 'block';
    document.getElementById('view-register').style.display = 'none';
    document.getElementById('view-main').style.display = 'none';
}

// --- 4. 业务逻辑 (已改造为 authFetch) ---

async function loadCollections() {
    try {
        const res = await authFetch(`${SERVER_URL}/api/collections`, { method: 'GET' });
        if(!res) return;
        const list = await res.json();
        
        const sel = document.getElementById('sel-collection');
        sel.innerHTML = '';
        list.forEach(c => {
            const opt = document.createElement('option');
            opt.value = c.id;
            opt.innerText = c.name;
            sel.appendChild(opt);
        });

        const lastColId = localStorage.getItem('wonderLastCollectionId');
        if (lastColId && list.some(c => c.id === lastColId)) {
            sel.value = lastColId;
        }

        if(sel.value) fetchCollectionList(true);
    } catch(e) {}
}

async function createCollection() {
    const name = prompt("新集合名称:");
    if(!name) return;
    
    // 注意：不需要传 username，后端从 Token 取
    await authFetch(`${SERVER_URL}/api/collections`, {
        method: 'POST',
        body: JSON.stringify({ name }) 
    });
    loadCollections();
}

async function handleSave() {
    if(!currentUid) detectPlatform();

    const payload = {
        influencerName: document.getElementById('inp-influencer').value,
        email: document.getElementById('inp-email').value,
        collectionId: document.getElementById('sel-collection').value,
        url: location.href,
        uid: currentUid, 
        platform: currentPlatform
        // 注意：这里不需要 username 了
    };
    
    if(!payload.collectionId) return alert('请选择集合');

    const btn = document.getElementById('btn-save');
    btn.innerText = '保存中...';
    try {
        const res = await authFetch(`${SERVER_URL}/api/leads`, {
            method: 'POST',
            body: JSON.stringify(payload)
        });
        if(!res) return;

        const data = await res.json();
        if(data.success) {
            document.getElementById('msg-box').innerText = '✅ Saved!';
            setTimeout(() => document.getElementById('msg-box').innerText = '', 2000);
            fetchCollectionList(true); 
        }
    } catch(e) { alert('保存失败'); }
    finally { btn.innerText = '保存线索'; }
}

async function fetchCollectionList(silent = false) {
    const colId = document.getElementById('sel-collection').value;
    if(!colId) return;
    
    if(!silent) document.getElementById('col-count-display').innerText = '刷新中...';

    try {
        const res = await authFetch(`${SERVER_URL}/api/leads?collectionId=${colId}`, { method: 'GET' });
        if(!res) return;

        const list = await res.json();
        currentCollectionList = list;
        
        document.getElementById('col-count-display').innerText = `当前集合: ${list.length} 条`;

        if(!silent) {
            document.getElementById('inp-list-search').value = '';
            triggerFilter(); 
        }
        checkDuplicate(); 
    } catch(e) {
        document.getElementById('col-count-display').innerText = '获取失败';
    }
}

// 统一的筛选触发器
function triggerFilter() {
    const keyword = document.getElementById('inp-list-search').value.toLowerCase().trim();
    filterList(keyword);
}

function filterList(keyword) {
    const onlyToday = document.getElementById('chk-today-only').checked;
    const onlyUnsent = document.getElementById('chk-unsent-only').checked;
    
    const now = new Date();
    const currentYear = now.getFullYear();
    const currentMonth = now.getMonth();
    const currentDay = now.getDate();

    filteredList = currentCollectionList.filter(item => {
        const matchKeyword = !keyword || (
            (item.influencer_name && item.influencer_name.toLowerCase().includes(keyword)) ||
            (item.uid && item.uid.includes(keyword))
        );

        const matchUnsent = !onlyUnsent || (item.status !== 'sent');

        let matchToday = true;
        if (onlyToday) {
            if (!item.created_at) {
                matchToday = false; 
            } else {
                const itemDate = new Date(item.created_at);
                matchToday = (itemDate.getFullYear() === currentYear &&
                              itemDate.getMonth() === currentMonth &&
                              itemDate.getDate() === currentDay);
            }
        }

        return matchKeyword && matchUnsent && matchToday;
    });

    renderList(filteredList);
    updateStats(currentCollectionList);
}

function renderList(list) {
    const container = document.getElementById('lead-list');
    document.getElementById('list-count').innerText = `共 ${list.length} 人`;

    container.innerHTML = '';
    
    if(list.length === 0) {
        container.innerHTML = '<div style="text-align:center; color:#999; margin-top:20px;">暂无数据</div>';
        return;
    }

    list.forEach(item => {
        const isSent = false; 
        
        const div = document.createElement('div');
        div.className = 'lead-item';
        
        const leftDiv = document.createElement('div');
        leftDiv.className = 'lead-left';
        leftDiv.innerHTML = `
            <div class="lead-name" title="${item.influencer_name}">${item.influencer_name}</div>
            <div class="lead-meta">UID: ${item.uid || '-'}</div>
            ${item.email ? `<div class="lead-meta lead-email">✉️ ${item.email}</div>` : ''}
        `;

        const rightDiv = document.createElement('div');
        rightDiv.className = 'lead-right';
        
        const statusSpan = document.createElement('span');
        statusSpan.className = `lead-status ${isSent ? 'sent' : ''}`;
        statusSpan.innerText = isSent ? '(已发信)' : '(未发信)';
        
        const delBtn = document.createElement('div');
        
        if (item.id) {
            delBtn.className = 'btn-delete';
            delBtn.innerHTML = '🗑️';
            delBtn.title = "删除记录";
            delBtn.onclick = (e) => {
                e.stopPropagation();
                handleDeleteLead(item.id);
            };
        } else {
            delBtn.className = 'btn-delete disabled';
            delBtn.innerHTML = '🚫';
            delBtn.title = "ID缺失";
        }

        rightDiv.appendChild(statusSpan);
        rightDiv.appendChild(delBtn);

        div.appendChild(leftDiv);
        div.appendChild(rightDiv);
        container.appendChild(div);
    });
}

async function handleDeleteLead(leadId) {
    if(!confirm('确定要删除这条记录吗？')) return;
    try {
        const res = await authFetch(`${SERVER_URL}/api/leads/${leadId}`, {
            method: 'DELETE'
        });
        if(!res) return;

        if(res.ok) {
            currentCollectionList = currentCollectionList.filter(i => i.id !== leadId);
            document.getElementById('col-count-display').innerText = `当前集合: ${currentCollectionList.length} 条`;
            triggerFilter();
        } else {
            alert('删除失败');
        }
    } catch(e) {
        alert('删除请求错误');
    }
}

function updateStats(fullList) {
    const stats = { TikTok: 0, Instagram: 0, YouTube: 0, Other: 0 };
    fullList.forEach(item => {
        if(stats[item.platform] !== undefined) stats[item.platform]++;
        else stats['Other']++;
    });
    const footer = document.getElementById('list-footer');
    footer.innerHTML = `
        <div class="stat-item"><span class="dot" style="background:#000;"></span> TT ${stats.TikTok}</div>
        <div class="stat-item"><span class="dot" style="background:#bc1888;"></span> IG ${stats.Instagram}</div>
        <div class="stat-item"><span class="dot" style="background:#f00;"></span> YTB ${stats.YouTube}</div>
    `;
}

function handleExportCsv() {
    if(!filteredList || filteredList.length === 0) return alert('当前列表为空 (请检查筛选条件)');
    
    const headers = ["Influencer Name", "Platform", "Email", "UID", "Status", "Link", "Created At"];
    const rows = filteredList.map(item => [
        item.influencer_name,
        item.platform,
        item.email || '',
        item.uid || '',
        item.status === 'sent' ? 'Sent' : 'Unsent',
        item.source_url || '',
        item.created_at ? new Date(item.created_at).toLocaleString() : ''
    ]);
    
    let csvContent = "data:text/csv;charset=utf-8,\uFEFF"; 
    csvContent += headers.join(",") + "\r\n";
    rows.forEach(row => {
        const rowStr = row.map(field => `"${String(field).replace(/"/g, '""')}"`).join(",");
        csvContent += rowStr + "\r\n";
    });

    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    const dateStr = new Date().toISOString().slice(0,10);
    
    const suffix = document.getElementById('chk-today-only').checked ? '_today' : '_all';
    
    link.setAttribute("download", `leadtracking_${dateStr}${suffix}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function checkDuplicate() {
    const currentName = document.getElementById('inp-influencer').value.trim();
    const warning = document.getElementById('dup-warning');
    if(!currentName) return;

    const exists = currentCollectionList.some(item => 
        item.influencer_name.toLowerCase() === currentName.toLowerCase() || 
        (currentUid && item.uid === currentUid)
    );
    warning.style.display = exists ? 'inline' : 'none';
}

async function handleBulkImport() {
    const text = document.getElementById('inp-bulk').value;
    if(!text.trim()) return;
    
    const colSelect = document.getElementById('sel-collection');
    const colId = colSelect.value;
    
    if (!colId) {
        return alert('❌ 导入失败：请先在下拉框中选择一个集合！');
    }
    
    const existingSet = new Set(
        currentCollectionList.map(item => item.influencer_name.toLowerCase().trim())
    );

    const lines = text.split('\n');
    const leads = [];
    let skipCount = 0;

    lines.forEach(line => {
        const parts = line.split(/[,\t]/);
        const name = parts[0]?.trim();
        const email = parts[1]?.trim();
        
        if(name) {
            if (existingSet.has(name.toLowerCase())) {
                skipCount++;
                return; 
            }

            leads.push({
                influencer_name: name,
                email: email,
                collection_id: colId, 
                platform: currentPlatform,
                source_url: 'Imported',
                uid: null
            });
            
            existingSet.add(name.toLowerCase());
        }
    });

    if(leads.length === 0) {
        if (skipCount > 0) return alert(`所有 ${skipCount} 条数据均已存在，无需导入。`);
        return;
    }

    const btn = document.getElementById('btn-do-import');
    btn.innerText = '导入中...';
    try {
        // 注意：bulk import 也不需要 username，后端Token取
        const res = await authFetch(`${SERVER_URL}/api/leads/bulk`, {
            method: 'POST',
            body: JSON.stringify({ leads }) 
        });
        
        if (res) {
            document.getElementById('inp-bulk').value = '';
            fetchCollectionList(); 
            alert(`✅ 成功导入 ${leads.length} 条 (自动跳过 ${skipCount} 条重复)`);
        }

    } catch(e) { 
        console.error(e);
        alert('导入失败，请检查网络或后端日志'); 
    }
    finally { btn.innerText = '执行导入'; }
}